# CSS Features Documentation

## Overview
This document details all CSS features used in The Soul Cats Band website.

## 1. CSS Reset
**File:** style.css (Lines 1-65)
- Meyer Web CSS Reset v2.0
- `box-sizing: border-box` applied to all elements
- Removes default margins, padding, and borders
- Resets typography and list styles

```css
*,
*:before,
*:after {
  box-sizing: border-box;
}
```

## 2. CSS Custom Properties (Variables)
**File:** style.css (Lines 70-100)

### Color Variables
```css
--primary-dark: #1a1a1a;
--primary-accent: #d4a574;
--primary-gold: #c9a961;
--secondary-dark: #2d2d2d;
--accent-light: #f5e6d3;
--text-dark: #333;
--text-light: #666;
--white: #ffffff;
```

### Typography Variables
```css
--font-display: 'Playfair Display', serif;
--font-body: 'Lato', sans-serif;
```

### Spacing Variables
```css
--spacing-xs: 0.5rem;
--spacing-sm: 1rem;
--spacing-md: 1.5rem;
--spacing-lg: 2rem;
--spacing-xl: 3rem;
```

### Transition Variables
```css
--transition-smooth: 0.3s ease-in-out;
--transition-fast: 0.15s ease;
```

## 3. CSS Grid Layouts

### Auto-fit Responsive Grid
```css
.grid {
	display: grid;
	gap: var(--spacing-lg);
	grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
}
```
- Used for: Event types, benefit cards
- Automatically creates columns based on available space
- Minimum column width: 280px
- Equal-width columns with gaps

### 2-Column Grid
```css
.grid-2 {
	display: grid;
	gap: var(--spacing-lg);
	grid-template-columns: repeat(2, 1fr);
}
```
- Used for: Footer, philosophy sections
- Fixed 2-column layout on desktop

### 3-Column Grid
```css
.grid-3 {
	display: grid;
	gap: var(--spacing-lg);
	grid-template-columns: repeat(3, 1fr);
}
```
- Used for: Band members, event types
- Fixed 3-column layout on desktop

### Specialized Grids
**Contact Container (2-column on desktop):**
```css
.contact-container {
	display: grid;
	grid-template-columns: 1fr 1fr;
	gap: var(--spacing-xl);
}
```

**Video Grid (responsive):**
```css
.video-grid {
	display: grid;
	gap: var(--spacing-lg);
	grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
}
```

## 4. Flexbox Layouts

### Primary Flex Container
```css
.flex {
	display: flex;
	gap: var(--spacing-md);
	align-items: center;
}
```
- Used for: Spacing elements with gaps
- Centered alignment

### Flex Utilities
```css
.flex-column { flex-direction: column; }
.flex-between { justify-content: space-between; }
.flex-center { justify-content: center; }
.flex-wrap { flex-wrap: wrap; }
```

### Header Navigation
```css
nav ul {
	display: flex;
	gap: var(--spacing-lg);
	list-style: none;
}
```

### Hero Section
```css
.hero {
	display: flex;
	flex-direction: column;
	justify-content: center;
	align-items: center;
}
```

### Social Links (Circular layout)
```css
.social-links {
	display: flex;
	gap: var(--spacing-md);
	justify-content: center;
}
```

## 5. CSS Transitions

### Smooth Transitions (0.3s)
```css
--transition-smooth: 0.3s ease-in-out;
```
Applied to:
- Link color changes
- Button hover effects
- Card hover transforms
- Form focus effects
- Navigation hover states

### Fast Transitions (0.15s)
```css
--transition-fast: 0.15s ease;
```
Applied to:
- Form input interactions
- Quick feedback effects

### Examples
**Link Hover:**
```css
a {
	transition: color var(--transition-smooth);
}
```

**Card Hover:**
```css
.card:hover {
	box-shadow: 0 8px 20px rgba(0, 0, 0, 0.15);
	transform: translateY(-4px);
	transition: all var(--transition-smooth);
}
```

**Button Hover:**
```css
.cta-button:hover {
	background-color: var(--primary-gold);
	transform: translateY(-2px);
	box-shadow: 0 4px 12px rgba(212, 165, 116, 0.3);
	transition: all var(--transition-smooth);
}
```

**Form Focus:**
```css
.form-group input:focus {
	outline: none;
	border-color: var(--primary-accent);
	box-shadow: 0 0 0 3px rgba(212, 165, 116, 0.1);
	transition: border-color var(--transition-smooth);
}
```

## 6. CSS Animations

### Fade In Down Animation
```css
@keyframes fadeInDown {
	from {
		opacity: 0;
		transform: translateY(-30px);
	}
	to {
		opacity: 1;
		transform: translateY(0);
	}
}
```
- Applied to: Hero h1
- Duration: 0.8s ease-out

### Fade In Up Animation
```css
@keyframes fadeInUp {
	from {
		opacity: 0;
		transform: translateY(30px);
	}
	to {
		opacity: 1;
		transform: translateY(0);
	}
}
```
- Applied to: Hero p (0.2s delay)
- Applied to: CTA button (0.4s delay)
- Creates staggered animation effect

## 7. Box Shadow Effects

### Subtle Shadow (Cards)
```css
box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
```

### Elevated Shadow (Hover)
```css
box-shadow: 0 8px 20px rgba(0, 0, 0, 0.15);
```

### Accent Shadow (Buttons)
```css
box-shadow: 0 4px 12px rgba(212, 165, 116, 0.3);
```

### Focus Shadow (Forms)
```css
box-shadow: 0 0 0 3px rgba(212, 165, 116, 0.1);
```

## 8. CSS Gradients

### Hero Overlay
```css
background: linear-gradient(135deg, rgba(26, 26, 26, 0.7), rgba(45, 45, 45, 0.7))
```

### Stat Box Gradient
```css
background: linear-gradient(135deg, var(--primary-accent), var(--primary-gold));
```

## 9. Color and Typography

### Font Stack
- Display: 'Playfair Display' (Google Fonts) - serif, elegant
- Body: 'Lato' (Google Fonts) - sans-serif, readable

### Color Scheme
- Primary Dark: #1a1a1a (jazz club feel)
- Accent Gold: #d4a574 (brass/soul music theme)
- Accent Light: #f5e6d3 (cream, readable text)
- Professional gray: #666 (secondary text)

### Hover States
```css
a:hover {
	color: var(--primary-gold);
}
```

## 10. Responsive Media Queries

### Tablet Breakpoint (411-1023px)
```css
@media (max-width: 1023px) {
	.grid-3 {
		grid-template-columns: repeat(2, 1fr);
	}
	.contact-container {
		grid-template-columns: 1fr;
	}
}
```

### Mobile Breakpoint (≤410px)
```css
@media (max-width: 410px) {
	nav {
		display: none;
	}
	
	nav.active {
		display: block;
		width: 100%;
	}
	
	.grid, .grid-2, .grid-3 {
		grid-template-columns: 1fr;
	}
	
	.menu-toggle {
		display: block;
	}
}
```

## 11. Special Form Styling

### Required Field Styling
```css
.form-group.required input,
.form-group.required textarea {
	border-left: 4px solid var(--error);
}

.required-asterisk {
	color: var(--error);
	font-weight: 700;
}
```

### Input Focus Styling
```css
.form-group input:focus,
.form-group textarea:focus {
	outline: none;
	border-color: var(--primary-accent);
	box-shadow: 0 0 0 3px rgba(212, 165, 116, 0.1);
}
```

## 12. Navigation Styling

### Active Link Indicator
```css
nav a.active {
	color: var(--primary-accent);
	border-bottom-color: var(--primary-accent);
}
```

### Mobile Menu Toggle
```css
.menu-toggle {
	display: none;
	background: none;
	border: none;
	color: var(--white);
	font-size: 1.5rem;
	cursor: pointer;
	padding: var(--spacing-sm);
}

@media (max-width: 410px) {
	.menu-toggle {
		display: block;
	}
}
```

## 13. Profile Card Styling

### Profile Cards
```css
.profile-card {
	background-color: var(--white);
	border-radius: 8px;
	overflow: hidden;
	box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
	display: flex;
	flex-direction: column;
}

.profile-card:hover {
	box-shadow: 0 12px 24px rgba(0, 0, 0, 0.15);
	transform: translateY(-6px);
}
```

## 14. Footer Styling

### Footer Grid Layout
```css
.footer-content {
	display: grid;
	grid-template-columns: repeat(3, 1fr);
	gap: var(--spacing-lg);
}

@media (max-width: 1023px) {
	.footer-content {
		grid-template-columns: 1fr;
		text-align: center;
	}
}
```

## 15. Utility Classes

### Spacing Utilities
```css
.mt-1, .mt-2, .mt-3, .mt-4 { margin-top: ... }
.mb-1, .mb-2, .mb-3, .mb-4 { margin-bottom: ... }
.pt-1, .pt-2, .pt-3, .pt-4 { padding-top: ... }
.pb-1, .pb-2, .pb-3, .pb-4 { padding-bottom: ... }
```

### Display Utilities
```css
.hidden { display: none !important; }
.visible { display: block !important; }
.text-center { text-align: center; }
.text-light { color: var(--text-light); }
```

## CSS Best Practices Used

1. **CSS Custom Properties** - Easy to maintain and update
2. **Mobile-First Approach** - Base styles for mobile, then enhance
3. **Semantic Class Names** - Descriptive, easy to understand
4. **Proper Specificity** - Avoiding !important except for utilities
5. **DRY Principle** - Reusable components and classes
6. **Performance** - Efficient selectors, minimal specificity
7. **Accessibility** - Color contrast, focus states, readable fonts
8. **Consistency** - Unified spacing, colors, and transitions

---

**Summary:** This website uses modern CSS techniques including Grid, Flexbox, Custom Properties, Animations, and Responsive Design to create a professional, scalable, and maintainable styling system.
